### Read in reconstruction results, generate ensemble median fields and confidence intervals and make Figure 1 (proxy map). RN May 2019

#required functions--------------
source("R-Functions_CFR.R")

#required packages---------------
require(RNetCDF)
require(fields)
library(rgdal)
library(ggplot2)
library(RColorBrewer)
library(scales)
library(parallel)

### definitions -------------------------------

#reference period
ref.start<-1911
ref.end<-1995

#start and year of the instrumental grid
instr.start<-1850


#file reading parameters
methods<-c("CPS","PCR","CCA","GraphEM","AM","DA")

recon.folder<-"recons/"

#variables in nc files
nc.vars.methods<-c(4,4,4,4,4,4)
nc.vars.lon.methods<-c(0,0,0,0,0,0)
nc.vars.lat.methods<-c(1,1,1,1,1,1)
nc.vars.ens.methods<-c(2,2,2,2,2,2)
nc.vars.times.methods<-c(3,3,3,3,3,3)

nexp<-length(methods)
ens.recon<-rep(T,nexp)
nens<-rep(100,nexp)

#number of cores for parallel computations
ncores<-20

#define colors for methods -----
#need to use colorblind compatible colors
exp.cols<-colblind.cols.tol[c(8,1,2,7,5,4)]
exp.cols.members<-rep(exp.cols,each=nens.do)
#show_col(exp.cols)


# read in instrumental  data- ---------
instrfile<-"input_data/HadCRUT4.3_GraphEM_SP80_18502014_Apr-Mar_corr.nc"
nc<-open.nc(instrfile)  
lats<-var.get.nc(nc,1)
lons<-var.get.nc(nc,0)
instr<-var.get.nc(nc,3)
close.nc(nc)
lons.2d<-as.vector(lons)
lats.2d<-as.vector(lats)
lons.grid<-lons
lats.grid<-lats
lons<-unique(lons.2d)
lats<-unique(lats.2d)

nlon<-length(lons)
nlat<-length(lats)
ncells<-length(lons.2d)

#anomalies
instr<-aperm(apply(instr,c(1,2),function(x) x-mean(x[(ref.start-instr.start+1):(ref.end-instr.start+1)])),c(2,3,1))


#field projections --------
lonsx<-lons
lonsx[which(lons>180)]<-lons[which(lons>180)]-360

center.lon<-(-120)

#world map
cl110<-readOGR("input_data/ne_110m_coastline", layer="ne_110m_coastline")

#project the field for the maps into robinson projection
field.proj<-project.field(instr[,,1],lonsx,lats,cl110,proj="robin",center=center.lon)

#project grid points
gridpoints.proj<-project(cbind(as.vector(lons.2d),as.vector(lats.2d)),paste0("+proj=robin +lon_0=",center.lon))


#generate coordinates for the x-axes on plots
lonsmapgrid.signs<-c(0,60,120,180,-120,-60,0,60)
lonsmapgrid.signs.deg<-paste0(c(0,60,120,180,120,60,0,60),"°",c(rep("E",4),rep("W",2),rep("E",2)))
lon.ax.fields<-c(field.proj$lonaxat,project(cbind(60,-90),paste0("+proj=robin +lon_0=",center.lon))[1])


# #get proxy metadata -----------
#fdr set
proxy.metadata.fdr<-as.matrix(read.table("input_data/metadata_2.0.0_calib-selection_1881_1916_1995_0.67_infilled_DINEOF_PAGES-crit-regional+FDR.txt",sep="\t"))
proxy.metadata.fdr<-proxy.metadata.fdr[,-1]
lons.proxies<-as.numeric(proxy.metadata.fdr[3,])
lons.proxies[which(lons.proxies<0)]<-lons.proxies[which(lons.proxies<0)]+360
lats.proxies<-as.numeric(proxy.metadata.fdr[2,])
coord.proxies<-cbind(lons.proxies,lats.proxies)

#project proxy coordinates
prox.cors.proj<-project(cbind(lons.proxies,lats.proxies),paste0("+proj=robin +lon_0=",center.lon))

##save general variables
#save(nexp,exp.cols,methods,nlon,nlat,ncells,field.proj,prox.cors.proj,
#     lons,lats,lonsx,lonsmapgrid.signs,lonsmapgrid.signs.deg,lon.ax.fields,
#     gridpoints.proj,file="general_definitions.RData")

# read in recon data and generate ensemble medians and conf. range------------------------

cl <- makeCluster(ncores)


fields<-list()
unc.l<-list()
unc.u<-list()

ntime<-2000

for(i in seq_along(methods)){
  #get filenames
  fname<-paste0(recon.folder,methods[i],".nc")
  nc<-open.nc(fname)
  nmembers<-length(var.get.nc(nc,nc.vars.ens.methods[i]))
  ensx<-var.get.nc(nc,nc.vars.methods[i],c(1,1,1,1),c(nlon,nlat,nmembers,ntime))
  x<-parApply(cl,ensx,c(1,2,4),quantile,probs=c(0.5,0.05,0.95))
  fields[[i]]<-x[1,,,]
  unc.l[[i]]<-x[2,,,]
  unc.u[[i]]<-x[3,,,]
  gc()
  close.nc(nc)
  print(i)
} 

gc()


#calculate anomalies ---------------------
for(i in seq_along(methods)){
  xd<-apply(fields[[i]],c(1,2),function(x) mean(x[ref.start:ref.end]))
  fields[[i]]<-fields[[i]]-array(xd,dim=c(72,36,dim(fields[[i]])[3]))
  if(length(unc.u[[i]])>0){
    unc.l[[i]]<-unc.l[[i]]-array(xd,dim=c(72,36,dim(fields[[i]])[3]))
    unc.u[[i]]<-unc.u[[i]]-array(xd,dim=c(72,36,dim(fields[[i]])[3]))
  }
}

# save recon data workspace-----------------
#save(fields,file="fields1.RData")
#save(unc.l,unc.u,file="UQ.RData")


#width of uncertainty estimates------------------------
sd.instr<-apply(instr[,,62:146],c(1,2),sd)
ci.widths.rel<-c()
for(exp in 1:nexp){
  ci.w<-unc.u[[exp]]-unc.l[[exp]]
  ci.w.rel<-array(apply(ci.w,3,function(x) x/sd.instr),dim=c(dim(ci.w)[1],dim(ci.w)[2],dim(ci.w)[3]))
  ci.widths.rel<-cbind(ci.widths.rel,ts(latweightmean.field(ci.w.rel,lats = lats),start=1))
  print(exp)
}

ci.widths.rel.mean<-tsapply(ci.widths.rel,1,mean)


## calculations for Fig 1.---------------------------------------
#proxy-gridpoint distances
mindist<-array(dim=c(nlon,nlat))

for(lo in 1:nlon){
  for(la in 1:nlat){
    dists<-apply(coord.proxies,1,function(x) distanz(x[2],x[1],lats[la],lons[lo]))
    mindist[lo,la]<-min(dists)
  }
  print(lo)
}

#archive types
archnames<-sort(unique(proxy.metadata.fdr[4,]))
archx<-match(proxy.metadata.fdr[4,],archnames)

#use archive color and point scheme formn PAGES 2k paper
archcols<-c(rgb(1,0.839843750000000,0),
            rgb(0.738281250000000,0.714843750000000,0.417968750000000),
            rgb(1,0.546875000000000,0),
            rgb(0,0,0),
            rgb(0.527343750000000,0.804687500000000,0.979166666666667),
            rgb(0,0.746093750000000,1),
            rgb(0.253906250000000,0.410156250000000,0.878906250000000),
            rgb(0.542968750000000,0.269531250000000,0.0742187500000000),
            rgb(1,0,0),
            rgb(1,0.0781250000000000,0.574218750000000),
            rgb(0.195312500000000,0.800781250000000,0.195312500000000))

archsymbs<-c(-as.hexmode(10017),11,21,-as.hexmode(9733),23,8,22,22,21,23,24)

#make hybrid black and last
archcols[6]<-1
#make bivalve different symbol and color
archcols[1]<-"purple"
archsymbs[1]<-21
pointsorder<-c((1:length(archx))[-which(archx==4 | archx ==1)],which(archx==4 | archx ==1))

#remove archives that are not in the subset used herein
archcols<-archcols[-c(2,4,8,9,10)]
archsymbs<-archsymbs[-c(2,4,8,9,10)]


save(proxy.metadata.fdr,lons.proxies,lats.proxies,coord.proxies,prox.cors.proj,
     mindist,archnames,archx,archcols,archsymbs,mindist,
     file="Proxy_locations_distances.RData")

#proxy availability over time
proxies<-read.ts("input_data/proxy_ama_2.0.0_calib-selection_1881_1916_1995_0.67_infilled_DINEOF_PAGES-crit-regional+FDR.txt",sep="\t",header=T)

archs.fdr<-sort(unique(archx))
nproxies.archs.fdr<-array(dim=c(2000,length(archs.fdr)))

for(ar in seq_along(archs.fdr)){
  selx<-which(archx==archs.fdr[ar])
  if(length(selx)>1){
    nproxies.archs.fdr[,ar]<-apply(window(proxies[,selx],1,2000),1,function(x) length(which(!is.na(x))))
  }else{
    nproxies.archs.fdr[,ar]<-window(proxies[,selx],1,2000)/window(proxies[,selx],1,2000)
  }
}
nproxies.archs.fdr[which(is.na(nproxies.archs.fdr))]<-0
nproxies.archs.fdr.totmax<-max(apply(nproxies.archs.fdr,1,sum))

nproxies.archs.fdr.cs<-nproxies.archs.fdr
nproxies.archs.fdr.cs<-t(apply(nproxies.archs.fdr.cs,1,cumsum))
nproxies.archs.fdr.cs<-ts(cbind(rep(0,2000),nproxies.archs.fdr.cs),start=1)

#colours for the map shading
range.mindist<-seq(0,max(mindist)+500,by=500)
cols.mindist<-brewer.pal(length(range.mindist)-1,"Blues")
cols.mindist<-(designer.colors(n=length(range.mindist)-1,col=cols.mindist))



# Make the Figure 1 ---------------------------------
ax1b<-"axis(1,at=lon.ax.fields[c(2,4,6,8)],labels=lonsmapgrid.signs.deg[c(2,4,6,8)],cex.axis=cext)"
layoutmatrix<-(cbind(c(0,0),
                     c(1,1),
                     c(0.3,0),
                     c(1,0.3)))
linescol<-"grey"
withincommands<-c("lines(field.proj$wm,lwd=2,col=1)",
                  "lines(field.proj$gridlines,col=linescol)")
cext<-1

latslab<-paste0(abs(field.proj$latsmapgrid),"°",c(rep("S",3),rep("N",4)))

pdf("Fig_1.pdf",width=7.2,height=5.76)
	split.screen(layoutmatrix)
	screen(1)
	par(mai=c(0.5,0.6,0.3,0.25))
	ax1<-"axis(1,at=lon.ax.fields,labels=lonsmapgrid.signs.deg,cex.axis=cext)"
	ax2<-"axis(2,at=field.proj$lataxat,labels=latslab,las=2,cex.axis=cext)"
	pts<-"points(prox.cors.proj[,1][pointsorder],prox.cors.proj[,2][pointsorder],col=archcols[archx][pointsorder],bg=archcols[archx][pointsorder],pch=archsymbs[archx][pointsorder])"
	image.plot3(field.proj$lons,field.proj$lats,mindist[field.proj$xo,],col.lab="white",
				col.axis="white",tck=0,col=cols.mindist,breaks = range.mindist,axis.args = list(cex.axis=cext),
				withincommands=c(withincommands,ax1,ax1b,ax2,pts))
	text(25300000,4007028,"Distance to closest proxy record [km]",srt=90,xpd=T,cex=cext)
	tw<-twf(archnames[archs.fdr],cex=cext)
	legend( -18143205,11300000,archnames[archs.fdr],pch=archsymbs[archs.fdr],col=archcols[archs.fdr],
			pt.bg=archcols[archs.fdr],bty="n",ncol=6,xpd=T,cex = cext,text.width = tw)

	screen(2)
	par(mai=c(0.8,0.9,0,0.8))
	plot(NA,xlim=c(1,2000),ylim=c(0,nproxies.archs.fdr.totmax),axes=F,ylab="",xlab="",cex.lab=cext)
	axis(1,at=seq(0,2000,by=200),labels=seq(0,2000,by=200),cex.axis=cext)
	axis(1,at=seq(0,2000,by=100),labels=NA)
	axis(2,las=2,cex.axis=cext)
	for(i in 1:length(archs.fdr)){
	  tspolygon(lower = nproxies.archs.fdr.cs[,i],upper = nproxies.archs.fdr.cs[,i+1],col=archcols[archs.fdr][i],border = archcols[archs.fdr][i])
	}
	par(new=T)
	plot(tsfilt(ci.widths.rel.mean,31,"bw"),xlim=c(1,2000),axes=F,xlab="",ylab="",lwd=2,col=2)
	axis(4,las=2,col.axis = 2,col.ticks = 2,col=2,cex.axis=cext)
	mtext(side=1,"Year CE",cex=cext,padj=5)
	mtext(side=2,"Number of records",cex=cext,padj=-4.2)
	mtext(side=4,"Mean CI width [Std.dev.]",padj=4.5,col=2,cex=cext)

	close.screen(all = T)
	par(new=T)
	par(mai=c(0,0,0,0))
	plot(1,t="n",axes=F,xlab="",ylab="",ylim=c(0,1),xlim=c(0,1))
	text(-0.02,1.02,"a",xpd=T,font=2,cex=cext)
	text(-0.02,0.35,"b",xpd=T,font=2,cex=cext)
dev.off()


###
stopCluster(cl)
